/*
 * Decompiled with CFR 0.152.
 */
package com.revrobotics.bootloader;

import com.revrobotics.bootloader.CRC32Mpeg2;
import com.revrobotics.canbridge.CanBridge;
import com.revrobotics.canbridge.CanBus;
import com.revrobotics.canbridge.CanMessage;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

public class REVUpBootloader {
    private final CanBridge bridge;
    private final CanBus bus;
    private final int deviceType;
    private final CRC32Mpeg2 crc = new CRC32Mpeg2();
    private volatile REVUpUpdateStatus status = REVUpUpdateStatus.IDLE;
    private volatile int percentage = 0;

    public REVUpBootloader(CanBridge bridge, CanBus bus, int deviceType) {
        this.bridge = bridge;
        this.bus = bus;
        this.deviceType = deviceType;
    }

    public void startFlashingApp(int canId, InputStream appBinary, int numRetries) {
        this.sendEnterBootloaderCommand(canId);
        this.initializeUpdateWithRetries(canId);
        this.sendBinary(canId, appBinary, numRetries);
        this.exitBootloader(canId);
        this.status = REVUpUpdateStatus.DONE;
    }

    private void sendEnterBootloaderCommand(int canId) {
        this.status = REVUpUpdateStatus.ENTERING_BOOTLOADER;
        int messageId = this.getHostArbId(this.deviceType, 1) | canId;
        boolean mode = false;
        byte[] data = new byte[]{7, -80, -83, 16, (byte)(mode ? 1 : 0), 0, 0, 0};
        CanMessage message = CanMessage.createClassic((int)messageId, (byte[])data);
        for (int i = 0; i < 3; ++i) {
            this.bridge.sendMessage(this.bus, message);
            REVUpBootloader.delay1ms();
        }
    }

    private void initializeUpdateWithRetries(int canId) {
        this.status = REVUpUpdateStatus.INITIALIZING;
        for (int i = 0; i < 3; ++i) {
            boolean success = this.initializeUpdate(canId, 5000);
            if (!success) continue;
            return;
        }
        this.status = REVUpUpdateStatus.FAILED;
        throw new IllegalStateException("failed to initialize bootloader");
    }

    private boolean initializeUpdate(int canId, int timeout) {
        int initCommandId = this.getHostArbId(this.deviceType, 3) | canId;
        int initKey = 49363;
        byte[] initData = new byte[8];
        initData[0] = (byte)(initKey & 0xFF);
        initData[1] = (byte)(initKey >> 8);
        CanMessage initMessage = CanMessage.createClassic((int)initCommandId, (byte[])initData);
        this.bridge.sendMessage(this.bus, initMessage);
        return this.waitForAck(canId, 3, timeout);
    }

    private boolean waitForAck(int canId, int commandIndex, int timeout) {
        for (int i = 0; i < timeout; ++i) {
            int initResponseId = this.getDeviceArbId(this.deviceType, 1) | canId;
            CanMessage initResponse = this.bridge.getLatestMessage(this.bus, initResponseId, 0x1FFFFFFF);
            if (initResponse != null && this.getAckCommand(initResponse.data()) == commandIndex) {
                return true;
            }
            REVUpBootloader.delay1ms();
        }
        return false;
    }

    private int getAckCommand(byte[] data) {
        if (data.length < 2) {
            System.err.println("Got invalid ack with length: " + data.length);
            return -1;
        }
        return data[0];
    }

    private void sendBinary(int canId, InputStream appBinary, int numRetries) {
        this.status = REVUpUpdateStatus.SENDING_BINARY;
        int address = 0;
        int lastPercentage = 0;
        try {
            int appSize = appBinary.available();
            int numTries = 0;
            int maxTries = 0;
            byte[] block = appBinary.readNBytes(64);
            while (block.length > 0) {
                boolean success = this.sendBlock(canId, block, address, 200);
                if (success) {
                    numTries = 0;
                    block = appBinary.readNBytes(64);
                    this.percentage = (address += block.length) * 100 / appSize;
                    if (this.percentage <= lastPercentage) continue;
                    System.out.println("Uploading: " + this.percentage + "%");
                    lastPercentage = this.percentage;
                    continue;
                }
                if (++numTries > maxTries) {
                    maxTries = numTries;
                }
                if (numTries < numRetries) continue;
                this.status = REVUpUpdateStatus.FAILED;
                System.out.println("On attempt number " + numTries + " for address " + address);
                throw new IllegalStateException("Timed out while sending the binary");
            }
            System.out.println("Uploaded successfully. The highest retry count was: " + maxTries);
        }
        catch (IOException e) {
            this.status = REVUpUpdateStatus.FAILED;
            System.err.println("Error reading app binary for device " + canId);
            throw new IllegalStateException("Error reading app binary for device " + canId, e);
        }
    }

    private boolean sendBlock(int canId, byte[] inBlock, int address, int timeout) throws IOException {
        byte[] block = new byte[64];
        System.arraycopy(inBlock, 0, block, 0, inBlock.length);
        this.crc.reset();
        this.crc.update(block);
        int responseId = this.getDeviceArbId(this.deviceType, 3) | canId;
        CanMessage oldResponseMessage = this.bridge.getLatestMessage(this.bus, responseId, 0x1FFFFFFF);
        this.startWritingWithRetries(canId, address, (int)this.crc.getValue());
        this.sendBlockData(canId, block);
        for (int i = 0; i < timeout; ++i) {
            boolean isOldMessage;
            CanMessage responseMessage = this.bridge.getLatestMessage(this.bus, responseId, 0x1FFFFFFF);
            boolean bl = isOldMessage = oldResponseMessage != null && oldResponseMessage.timestamp() == responseMessage.timestamp();
            if (responseMessage != null && !isOldMessage) {
                boolean matches;
                ByteBuffer buffer = ByteBuffer.wrap(responseMessage.data()).order(ByteOrder.LITTLE_ENDIAN);
                int responseAddress = buffer.getInt();
                long responseCrc = Integer.toUnsignedLong(buffer.getInt(4));
                boolean bl2 = matches = responseAddress == address && responseCrc == this.crc.getValue();
                if (matches) {
                    return true;
                }
                System.out.println("Address or CRC didn't match: " + responseAddress + " != " + address + " or " + responseCrc + " != " + this.crc.getValue());
                return false;
            }
            REVUpBootloader.delay1ms();
        }
        return false;
    }

    private synchronized void sendBlockData(int canId, byte[] blockData) {
        for (int i = 0; i < 8; ++i) {
            byte[] chunk = new byte[8];
            System.arraycopy(blockData, i * 8, chunk, 0, 8);
            int writeId = this.getHostArbId(this.deviceType, 6 + i) | canId;
            CanMessage message = CanMessage.createClassic((int)writeId, (byte[])chunk);
            this.bridge.sendMessage(this.bus, message);
            REVUpBootloader.delay1ms();
        }
    }

    private void startWritingWithRetries(int canId, int address, int crc) {
        for (int i = 0; i < 3; ++i) {
            boolean success = this.sendStartWrite(true, canId, address, crc, 100);
            if (!success) continue;
            return;
        }
        this.status = REVUpUpdateStatus.FAILED;
        throw new IllegalStateException("failed to initialize bootloader");
    }

    private boolean sendStartWrite(boolean isApp, int canId, int address, int crc, int timeout) {
        int apiIndex = isApp ? 4 : 5;
        int startWriteCanId = this.getHostArbId(this.deviceType, apiIndex) | canId;
        ByteBuffer buffer = ByteBuffer.allocate(8);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        buffer.putInt(address);
        buffer.putInt(crc);
        CanMessage message = CanMessage.createClassic((int)startWriteCanId, (byte[])buffer.array());
        this.bridge.sendMessage(this.bus, message);
        return this.waitForAck(canId, apiIndex, timeout);
    }

    private void exitBootloader(int canId) {
        this.status = REVUpUpdateStatus.EXITING_BOOTLOADER;
        int exitBootloaderId = this.getHostArbId(this.deviceType, 2) | canId;
        byte[] data = new byte[8];
        data[0] = 1;
        this.bridge.sendMessage(this.bus, CanMessage.createClassic((int)exitBootloaderId, (byte[])data));
    }

    private static void delay(int ms) {
        try {
            Thread.sleep(ms);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private static void delay1ms() {
        REVUpBootloader.delay(1);
    }

    private int getCrc(byte[] data) {
        this.crc.reset();
        this.crc.update(data);
        return (int)this.crc.getValue();
    }

    private int getHostArbId(int deviceType, int apiIndex) {
        return deviceType << 24 | 0x8000 | apiIndex << 6 | 0x50000;
    }

    private int getDeviceArbId(int deviceType, int apiIndex) {
        return deviceType << 24 | 0x8800 | apiIndex << 6 | 0x50000;
    }

    public REVUpUpdateStatus getStatus() {
        return this.status;
    }

    public int getPercentage() {
        return this.percentage;
    }

    public static enum REVUpUpdateStatus {
        FAILED,
        IDLE,
        ENTERING_BOOTLOADER,
        INITIALIZING,
        SENDING_BINARY,
        EXITING_BOOTLOADER,
        DONE;

    }
}

